import { Fragment, jsx, jsxs } from "react/jsx-runtime";
import {
  BaseBoxShapeUtil,
  FileHelpers,
  HTMLContainer,
  Image,
  MediaHelpers,
  Vec,
  fetch,
  imageShapeMigrations,
  imageShapeProps,
  resizeBox,
  structuredClone,
  toDomPrecision
} from "@tldraw/editor";
import classNames from "classnames";
import { useEffect, useState } from "react";
import { BrokenAssetIcon } from "../shared/BrokenAssetIcon.mjs";
import { HyperlinkButton } from "../shared/HyperlinkButton.mjs";
import { useAsset } from "../shared/useAsset.mjs";
import { usePrefersReducedMotion } from "../shared/usePrefersReducedMotion.mjs";
async function getDataURIFromURL(url) {
  const response = await fetch(url);
  const blob = await response.blob();
  return FileHelpers.blobToDataUrl(blob);
}
class ImageShapeUtil extends BaseBoxShapeUtil {
  static type = "image";
  static props = imageShapeProps;
  static migrations = imageShapeMigrations;
  isAspectRatioLocked = () => true;
  canCrop = () => true;
  getDefaultProps() {
    return {
      w: 100,
      h: 100,
      assetId: null,
      playing: true,
      url: "",
      crop: null,
      flipX: false,
      flipY: false
    };
  }
  onResize = (shape, info) => {
    let resized = resizeBox(shape, info);
    const { flipX, flipY } = info.initialShape.props;
    resized = {
      ...resized,
      props: {
        ...resized.props,
        flipX: info.scaleX < 0 !== flipX,
        flipY: info.scaleY < 0 !== flipY
      }
    };
    return resized;
  };
  isAnimated(shape) {
    const asset = shape.props.assetId ? this.editor.getAsset(shape.props.assetId) : void 0;
    if (!asset) return false;
    return "mimeType" in asset.props && MediaHelpers.isAnimatedImageType(asset?.props.mimeType) || "isAnimated" in asset.props && asset.props.isAnimated;
  }
  component(shape) {
    const isCropping = this.editor.getCroppingShapeId() === shape.id;
    const prefersReducedMotion = usePrefersReducedMotion();
    const [staticFrameSrc, setStaticFrameSrc] = useState("");
    const [loadedSrc, setLoadedSrc] = useState("");
    const isSelected = shape.id === this.editor.getOnlySelectedShapeId();
    const { asset, url } = useAsset(shape.id, shape.props.assetId, shape.props.w);
    useEffect(() => {
      if (url) {
        let cancelled = false;
        const image = Image();
        image.onload = () => {
          if (cancelled) return;
          setLoadedSrc(url);
        };
        image.src = url;
        return () => {
          cancelled = true;
        };
      }
    }, [url, shape]);
    useEffect(() => {
      if (url && this.isAnimated(shape)) {
        let cancelled = false;
        const image = Image();
        image.onload = () => {
          if (cancelled) return;
          const canvas = document.createElement("canvas");
          canvas.width = image.width;
          canvas.height = image.height;
          const ctx = canvas.getContext("2d");
          if (!ctx) return;
          ctx.drawImage(image, 0, 0);
          setStaticFrameSrc(canvas.toDataURL());
          setLoadedSrc(url);
        };
        image.crossOrigin = "anonymous";
        image.src = url;
        return () => {
          cancelled = true;
        };
      }
    }, [prefersReducedMotion, url, shape]);
    if (asset?.type === "bookmark") {
      throw Error("Bookmark assets can't be rendered as images");
    }
    const showCropPreview = isSelected && isCropping && this.editor.isIn("select.crop");
    const reduceMotion = prefersReducedMotion && (asset?.props.mimeType?.includes("video") || this.isAnimated(shape));
    const containerStyle = getCroppedContainerStyle(shape);
    if (!asset?.props.src) {
      return /* @__PURE__ */ jsxs(
        HTMLContainer,
        {
          id: shape.id,
          style: {
            overflow: "hidden",
            width: shape.props.w,
            height: shape.props.h,
            color: "var(--color-text-3)",
            backgroundColor: asset ? "transparent" : "var(--color-low)",
            border: asset ? "none" : "1px solid var(--color-low-border)"
          },
          children: [
            /* @__PURE__ */ jsx("div", { className: "tl-image-container", style: containerStyle, children: asset ? null : /* @__PURE__ */ jsx(BrokenAssetIcon, {}) }),
            "url" in shape.props && shape.props.url && /* @__PURE__ */ jsx(HyperlinkButton, { url: shape.props.url, zoomLevel: this.editor.getZoomLevel() })
          ]
        }
      );
    }
    if (!loadedSrc) return null;
    return /* @__PURE__ */ jsxs(Fragment, { children: [
      showCropPreview && /* @__PURE__ */ jsx("div", { style: containerStyle, children: /* @__PURE__ */ jsx(
        "img",
        {
          className: "tl-image",
          crossOrigin: this.isAnimated(shape) ? "anonymous" : void 0,
          src: !shape.props.playing || reduceMotion ? staticFrameSrc : loadedSrc,
          referrerPolicy: "strict-origin-when-cross-origin",
          style: {
            opacity: 0.1
          },
          draggable: false
        }
      ) }),
      /* @__PURE__ */ jsxs(
        HTMLContainer,
        {
          id: shape.id,
          style: { overflow: "hidden", width: shape.props.w, height: shape.props.h },
          children: [
            /* @__PURE__ */ jsxs("div", { className: "tl-image-container", style: containerStyle, children: [
              /* @__PURE__ */ jsx(
                "img",
                {
                  className: classNames("tl-image", {
                    "tl-flip-x": shape.props.flipX && !shape.props.flipY,
                    "tl-flip-y": shape.props.flipY && !shape.props.flipX,
                    "tl-flip-xy": shape.props.flipY && shape.props.flipX
                  }),
                  crossOrigin: this.isAnimated(shape) ? "anonymous" : void 0,
                  src: !shape.props.playing || reduceMotion ? staticFrameSrc : loadedSrc,
                  referrerPolicy: "strict-origin-when-cross-origin",
                  draggable: false
                }
              ),
              this.isAnimated(shape) && !shape.props.playing && /* @__PURE__ */ jsx("div", { className: "tl-image__tg", children: "GIF" })
            ] }),
            shape.props.url && /* @__PURE__ */ jsx(HyperlinkButton, { url: shape.props.url, zoomLevel: this.editor.getZoomLevel() })
          ]
        }
      )
    ] });
  }
  indicator(shape) {
    const isCropping = this.editor.getCroppingShapeId() === shape.id;
    if (isCropping) return null;
    return /* @__PURE__ */ jsx("rect", { width: toDomPrecision(shape.props.w), height: toDomPrecision(shape.props.h) });
  }
  async toSvg(shape) {
    if (!shape.props.assetId) return null;
    const asset = this.editor.getAsset(shape.props.assetId);
    if (!asset) return null;
    let src = await this.editor.resolveAssetUrl(shape.props.assetId, {
      shouldResolveToOriginal: true
    });
    if (!src) return null;
    if (src.startsWith("blob:") || src.startsWith("http") || src.startsWith("/") || src.startsWith("./")) {
      src = (await getDataURIFromURL(src)) || "";
    }
    const containerStyle = getCroppedContainerStyle(shape);
    const crop = shape.props.crop;
    if (containerStyle.transform && crop) {
      const { transform, width, height } = containerStyle;
      const croppedWidth = (crop.bottomRight.x - crop.topLeft.x) * width;
      const croppedHeight = (crop.bottomRight.y - crop.topLeft.y) * height;
      const points = [
        new Vec(0, 0),
        new Vec(croppedWidth, 0),
        new Vec(croppedWidth, croppedHeight),
        new Vec(0, croppedHeight)
      ];
      const cropClipId = `cropClipPath_${shape.id.replace(":", "_")}`;
      return /* @__PURE__ */ jsxs(Fragment, { children: [
        /* @__PURE__ */ jsx("defs", { children: /* @__PURE__ */ jsx("clipPath", { id: cropClipId, children: /* @__PURE__ */ jsx("polygon", { points: points.map((p) => `${p.x},${p.y}`).join(" ") }) }) }),
        /* @__PURE__ */ jsx("g", { clipPath: `url(#${cropClipId})`, children: /* @__PURE__ */ jsx("image", { href: src, width, height, style: { transform } }) })
      ] });
    } else {
      return /* @__PURE__ */ jsx("image", { href: src, width: shape.props.w, height: shape.props.h });
    }
  }
  onDoubleClick = (shape) => {
    const asset = shape.props.assetId ? this.editor.getAsset(shape.props.assetId) : void 0;
    if (!asset) return;
    const canPlay = asset.props.src && this.isAnimated(shape);
    if (!canPlay) return;
    this.editor.updateShapes([
      {
        type: "image",
        id: shape.id,
        props: {
          playing: !shape.props.playing
        }
      }
    ]);
  };
  onDoubleClickEdge = (shape) => {
    const props = shape.props;
    if (!props) return;
    if (this.editor.getCroppingShapeId() !== shape.id) {
      return;
    }
    const crop = structuredClone(props.crop) || {
      topLeft: { x: 0, y: 0 },
      bottomRight: { x: 1, y: 1 }
    };
    const w = 1 / (crop.bottomRight.x - crop.topLeft.x) * shape.props.w;
    const h = 1 / (crop.bottomRight.y - crop.topLeft.y) * shape.props.h;
    const pointDelta = new Vec(crop.topLeft.x * w, crop.topLeft.y * h).rot(shape.rotation);
    const partial = {
      id: shape.id,
      type: shape.type,
      x: shape.x - pointDelta.x,
      y: shape.y - pointDelta.y,
      props: {
        crop: {
          topLeft: { x: 0, y: 0 },
          bottomRight: { x: 1, y: 1 }
        },
        w,
        h
      }
    };
    this.editor.updateShapes([partial]);
  };
}
function getCroppedContainerStyle(shape) {
  const crop = shape.props.crop;
  const topLeft = crop?.topLeft;
  if (!topLeft) {
    return {
      width: shape.props.w,
      height: shape.props.h
    };
  }
  const w = 1 / (crop.bottomRight.x - crop.topLeft.x) * shape.props.w;
  const h = 1 / (crop.bottomRight.y - crop.topLeft.y) * shape.props.h;
  const offsetX = -topLeft.x * w;
  const offsetY = -topLeft.y * h;
  return {
    transform: `translate(${offsetX}px, ${offsetY}px)`,
    width: w,
    height: h
  };
}
export {
  ImageShapeUtil
};
//# sourceMappingURL=ImageShapeUtil.mjs.map
